###############################################################################
# cNTLogDataItemList
# class to store nt data columns
#-----------------------------------------------------------------------------#
# is needed here since cNTLogFileReader needs a list to generate the header in the datalog


###############################################################################
# cNTLogDataFrame
# this is the class to handle and host the data of one frame
#-----------------------------------------------------------------------------#
#this is base class
#class cNTDataFrameObject

#this is a child class for handling NT log files
#class cNTLogFileDataFrame(cNTDataFrameObject):

#this is a child class for handling NT serial data streams
#class cNTSerialDataFrame(cNTLogFileDataFrame):


###############################################################################
# cNTLogParser
# this is the main class to parse a stream of log packets into a cNTDataFrameObject
#-----------------------------------------------------------------------------#
#class cNTLogParser:


###############################################################################
# cNTLogFileReader
# this is the main class to read in a NTLogger data log file
# it generates a number of data lists, for easier handling in the GUI
#-----------------------------------------------------------------------------#
#class cNTLogFileReader:


import struct
import re
from math import sqrt, sin, pi
from PyQt5.QtCore import QFile
from owNTLoggerObjects_v039 import *


##defined already cNTLOGVERSION_UNINITIALIZED = 0
cLOGVERSION_NT_V2 = 2 #new V2 commands
cLOGVERSION_NT_V3 = 3 #SetLog extended to 0.001?
cLOGVERSION_NT_LATEST = 3 #this is an alias to the latest version


###############################################################################
# cNTLogDataItemList
# class to store nt data columns
#-----------------------------------------------------------------------------#
# is needed here since cNTLogFileReader needs a list to generate the header in the datalog
# the list must be compatible in sequence to the list used by cLogItemList in the main
class cNTLogDataItemList(cLogDataItemList):

    #this is a human-readable list of how to organize the standard NTLogger data field names into catagories
    #is used in getGraphSelectorList()

    def __init__(self,_translator=None):
        super().__init__(_translator)
        self.setToStandardNTLoggerItemList()  
        self.setToStandardNTLoggerGraphSelectorList()
        
    def setToStandardNTLoggerGraphSelectorList(self):
        self.graphSelectorList = [
        ['data1',["data1"] ],
        ['data2',["data2"] ],
        ['data3',["data3"] ],
        ['data4',["data4"] ],
        ['data5',["data5"] ],
        ['data6',["data6"] ],
        ]

    #the list order must be identical to that in class cNTLogFileReader !!!!
    #keeps information which is used for the various data formats
    def setToStandardNTLoggerItemList(self):
        self.clear()
        self.addItem( 'time', 'ms', cDATATYPE_U64, cDATATYPE_FLOAT )
        self.addItem( 'data1', 'int', cDATATYPE_U64, cDATATYPE_Ux )
        self.addItem( 'data2', 'int', cDATATYPE_U64, cDATATYPE_Ux )
        self.addItem( 'data3', 'int', cDATATYPE_U64, cDATATYPE_Ux )
        self.addItem( 'data4', 'int', cDATATYPE_U64, cDATATYPE_Ux )
        self.addItem( 'data5', 'int', cDATATYPE_U64, cDATATYPE_Ux )
        self.addItem( 'data6', 'int', cDATATYPE_U64, cDATATYPE_Ux )

    #self.graphSelectorList may be a larger list than that generated by getGraphSelectorList() 
    def getGraphSelectorDefaultIndex(self, graphSelectorList=None): #allows to check in a modified list
        if not graphSelectorList: graphSelectorList = self.graphSelectorList
        return 2
        return None
        

###############################################################################
# cNTLogDataFrame
# this is the class to handle and host the data of one frame
#-----------------------------------------------------------------------------#
# NTLogger decodes the data on the NT bus and stores that decoded data on the SD card
# error handling:
#   there are two types of error, (i) a package is incomplete, (ii) a crucial package is not complete
#   self.error is set, depending on the general error type
#   each doXXX returns True or False, so that a parser can determine more detailed error conditions

cNTDATAFRAME_OK = 0
cNTDATAFRAME_CMDERROR = 1
cNTDATAFRAME_SETMOTERROR = 2
cNTDATAFRAME_SETLOGERROR = 4

cNTBUS_MOTOR_FLAG_FOC = 0x20 #is needed to distinguish "non-encoder" and  "encoder" versions of SetMotAll

#this is a base class
# it holds all data, and provides convenience functions to set them, and extract them as dataline and rawline
class cNTDataFrameObject:

    def __init__(self):
        self.logVersion = cLOGVERSION_NT_LATEST #allows to detect different log file versions, latest version as default
        self.Time = 0 #that's the actual time of a data frame

        self.clear()
        
    def setLogVersion(self,ver):
        self.logVersion = ver

    def getLogVersion(self):
        return self.logVersion

    def clear(self):
        self.TimeStamp32 = 0
        self.data1 = 0
        self.data2 = 0
        self.data3 = 0
        self.data4 = 0
        self.data5 = 0
        self.data6 = 0
        self.error = cNTDATAFRAME_OK #new frame, new game

    #allows to do tests to check the integrety of teh data inthe frame, and to skip if invalid
    def isValid(self):
        return True 
        
    #------------------------------------------
    #NTbus data logs: the order must match that of setToDataLoggerItemList()
    def getDataLine(self):
        dataline = ''
        dataline +=  '{:.1f}'.format(0.001*self.Time) + "\t"
        dataline +=  str(self.data1) + "\t"
        dataline +=  str(self.data2) + "\t"
        dataline +=  str(self.data3) + "\t"
        dataline +=  str(self.data4) + "\t"
        dataline +=  str(self.data5) + "\t"
        dataline +=  str(self.data6) + "\n"
        return dataline

    #------------------------------------------
    #NTbus data logs: the order must match that of setToDataLoggerItemList()
    def getRawDataLine(self):
        rawdataline = []
        rawdataline.append(self.Time)
        rawdataline.append(self.data1)
        rawdataline.append(self.data2)
        rawdataline.append(self.data3)
        rawdataline.append(self.data4)
        rawdataline.append(self.data5)
        rawdataline.append(self.data6)
        return rawdataline

    #this corrects the time return by getDataLine() and getRawDataLine()
    def calculateTime(self,datalog_TimeStamp32_start):
        self.Time = self.TimeStamp32 - datalog_TimeStamp32_start

      

#this is a child class for handling NT log files
# it does the unpacking of the received payload bytes 
class cNTLogFileDataFrame(cNTDataFrameObject):

    def __init__(self):
        super().__init__()



#this is a child class for handling NT serial data streams
# it does the encoding of the data onto the NT bus
#the reader must provide a function
# reader.readPayload(length)
#
# a frame error must ONLY be thrown, then one of the crucial packages is wrong,
# i.e. SetLogger, SetMotorAll !!
class cNTSerialDataFrame(cNTLogFileDataFrame):

    def __init__(self, _reader):
        super().__init__()
        self.reader = _reader


###############################################################################
# cNTLogParser
# this is the main class to parse a stream of log packets into a cNTDataFrameObject
#-----------------------------------------------------------------------------#

#the reader must provide a function
# reader.appendDataFrame(frame)
#
# baseTime allows to shift the start time
class cNTLogParser:

    def __init__(self,_frame,_reader,_baseTime=0):
        self.reader = _reader

        self.frame = _frame
        self.frame.clear()
        self.line = ''

        #the TimeStamp32 CANNOT be 0, so 0 can also be used instead of -1 !! NOOO: when it's a serial stream it may start at 0
        self.startTimeStamp32 = None #also allows to detect that a first valid data frame was read
        self.baseTimeStamp32 = _baseTime #allows to shift the time axis
        
        self.reLine = re.compile(r'^[0-9.+-E \r\n\t,]+$')


    def clearForNextDataFrame(self):
        self.frame.clear()
        self.line = ''

    #------------------------------------------
    #get data from reader, and parse into a cNTLogDataFrameObhject()
    def parse(self,cmdid,cmdbyte=None,payload=None): #c = int(b[0])
        # /n = CR + LF = 13 + 10 
        if cmdid == 13:
            return False #no need to call analyzeAndAppend()
        if cmdid == 10:
            pass
        else:
            self.line += chr(cmdid)
        return True

    #------------------------------------------
    #analyzes the received frame, and appends it if correct
    # stxerror allows the caller to have additional errors considered
    # returns error
    def analyzeAndAppend(self,cmdid,stxerror): #c = int(b[0])
        frameError = False
        #self.reader.appendDataFrame(self.frame):
        if cmdid == 10: # /n = CR + LF = 13 + 10
            self.line = self.line.strip()
            d = self.line.split(',')
            if not self.reLine.search(self.line):
                frameError = True
            if len(d) != 8: 
                frameError = True
            
            if not frameError:
                self.frame.TimeStamp32 = int(d[0])*1000 #time in micros
                self.frame.data1 = float(d[1]) #int(d[1])
                self.frame.data2 = float(d[2]) #int(d[2])
                self.frame.data3 = float(d[3]) #int(d[3])
                self.frame.data4 = float(d[4]) #int(d[4])
                self.frame.data5 = float(d[5]) #int(d[5])
                self.frame.data6 = float(d[6]) #int(d[6])
            
                if self.startTimeStamp32 == None: 
                    self.startTimeStamp32 = self.frame.TimeStamp32
                self.frame.calculateTime( self.startTimeStamp32-self.baseTimeStamp32 )
                self.reader.appendDataFrame(self.frame)
                
            self.clearForNextDataFrame()
        return frameError


###############################################################################
# cNTLogFileReader
# this is the main class to read in a NTLogger data log file
# it generates a number of data lists, for easier handling in the GUI
#-----------------------------------------------------------------------------#
class cNTLogFileReader:

    def __init__(self):
        self.logVersion = cLOGVERSION_UNINITIALIZED #was a BUG, right? cLOGTYPE_UNINITIALIZED #private

    def readLogFile(self,loadLogThread,fileName,createTraffic):
        try:
            F = open(fileName, 'rb')
        except:
            return '','',''


        frame = cNTLogFileDataFrame()
        parser = cNTLogParser(frame, self)

        logItemList = cNTLogDataItemList()

        trafficlog = []

        #need to be self so that appendDataFrame() can be called by the parser
        self.datalog = []
        self.datalog.append( logItemList.getNamesAsStr('\t') + '\n' )
        self.datalog.append( logItemList.getUnitsAsStr('\t') + '\n' )

        self.rawdatalog = []

        byte_counter = 0
        byte_max = QFile(fileName).size()
        byte_percentage = 0
        byte_step = 5

        ##FBytesIO = BytesIO(F.read())  //THIS IS NOT FASTER AT ALL!!
        ##header = FBytesIO.read(9)
        ##payload = FBytesIO.read(size-9)
        frame.setLogVersion(cLOGVERSION_NT_V2) #assume <v0.03 as default
        while 1:
            if( loadLogThread.canceled ): break

            header = F.read(9)

        #end of while 1:
        F.close();
        loadLogThread.emitProgress(95)
        if( loadLogThread.canceled ):
            trafficlog = []
            self.datalog = []
            self.rawdatalog = []
        self.logVersion = frame.getLogVersion()
        return trafficlog, self.datalog, self.rawdatalog

    #this is called by the parser
    # returns a bool, True if error occured
    def appendDataFrame(self,_frame):
        self.datalog.append( _frame.getDataLine() )
        self.rawdatalog.append( _frame.getRawDataLine() )
        return False

    def getLogVersion(self):
        return self.logVersion
