# Version v0.41 08. June. 2018

###############################################################################
# cNTLogDataItemList
# class to store nt data columns
#-----------------------------------------------------------------------------#
# is needed here since cNTLogFileReader needs a list to generate the header in the datalog

###############################################################################
# cNTLogDataFrame
# this is the class to handle and host the data of one frame
#-----------------------------------------------------------------------------#

#this is a child class for handling NT log files
#class cNTLogFileDataFrame(cNTDataFrameObject):

#this is a child class for handling NT serial data streams
#class cNTSerialDataFrame(cNTLogFileDataFrame):

###############################################################################
# cNTLogParser
# this is the main class to parse a stream of log packets into a cNTDataFrameObject
#-----------------------------------------------------------------------------#

###############################################################################
# cNTLogFileReader
# this is the main class to read in a NTLogger data log file
# it generates a number of data lists, for easier handling in the GUI
#-----------------------------------------------------------------------------#


import struct
from math import sqrt, sin, cos, pi, atan2
from PyQt5.QtCore import QFile
from owNTLoggerObjects_v041 import * #cLogDataItemList


##defined alreadycNTLOGVERSION_UNINITIALIZED = 0
cLOGVERSION_NT_V2 = 2 #new V2 commands
cLOGVERSION_NT_V3 = 3 #SetLog extended to 0.001?
cLOGVERSION_NT_LATEST = 3 #this is an alias to the latest version


###############################################################################
# cNTLogDataItemList
# class to store nt data columns
#-----------------------------------------------------------------------------#
# is needed here since cNTLogFileReader needs a list to generate the header in the datalog
# the list must be compatible in sequence to the list used by cLogItemList in the main
class cNTLogDataItemList(cLogDataItemList):

    #this is a human-readable list of how to organize the standard NTLogger data field names into catagories
    #is used in getGraphSelectorList()

    def __init__(self,_translator=None):
        super().__init__(_translator)
        self.setToStandardNTLoggerItemList()  
        self.setToStandardNTLoggerGraphSelectorList()
        
    def setToStandardNTLoggerGraphSelectorList(self):
        self.graphSelectorList = [
        ['Performance',["Imu1rx","Imu1done","PIDdone","Motdone","Imu2rx","Imu2done","Logdone","Loopdone"] ],
        ['Imu1 Pitch,Roll,Yaw',["Imu1Pitch","Imu1Roll","Imu1Yaw"] ],
        ['Imu2 Pitch,Roll,Yaw',["Imu2Pitch","Imu2Roll","Imu2Yaw"] ],
        ['Encoder Pitch,Roll,Yaw',["EncPitch","EncRoll","EncYaw"] ],
        ['PID Pitch,Roll,Yaw',["PIDPitch","PIDRoll","PIDYaw"] ],
        ['Ahrs1',["Rx1","Ry1","Rz1","AccAmp1","AccConf1","YawTarget2"]],
        ['State',['State']],
        ['Error',['ErrorCnt']],
        ['Voltage',['Voltage']],
        ['Acc1',["ax1","ay1","az1"]],
        ['Gyro1',["gx1","gy1","gz1"]],
        ['Acc2',["ax2","ay2","az2"]],
        ['Gyro2',["gx2","gy2","gz2"]],
        ['Sensor States',["Imu1State","Imu2State","MotState"]],
#STL        
        ['STorM32 Link',["q0","q1","q2","q3","vx","vy","vz","YawRateCmd","FCStatus","SLStatus"]],
        ['Acc1 raw',["ax1raw","ay1raw","az1raw"]],
        ['Gyro1 raw',["gx1raw","gy1raw","gz1raw"]],
        ['Acc2 raw',["ax2raw","ay2raw","az2raw"]],
        ['Gyro2 raw',["gx2raw","gy2raw","gz2raw"]],
        ['Acc3 raw',["ax3raw","ay3raw","az3raw"]],
        ['Gyro3 raw',["gx3raw","gy3raw","gz3raw"]],
        ['Temp 1+2+3',["T1","T2","T3"]],
        ['Encoder raw',["EncRawPitch","EncRawRoll","EncRawYaw"] ],
        ['dU raw',["dURawPitch","dURawRoll","dURawYaw"]],
        ['PID Mot Pitch,Roll,Yaw',["PIDMotPitch","PIDMotRoll","PIDMotYaw"]],
        ['Mot Flags',["MotFlags"]],
        ['Mot Pitch,Roll,Yaw',["MotPitch","MotRoll","MotYaw"]],
        ['Vmax Pitch,Roll,Yaw',["VmaxPitch","VmaxRoll","VmaxYaw"]],
        ['dU Pitch,Roll,Yaw',["dUPitch","dURoll","dUYaw"]],
        ['Camera',["CameraCmd","CameraPwm"]],
#STL        
        ['Debug',["debug1","debug2","debug3","debug4","debug5","debug6","debug7","inj1","inj2","inj3"]],
        ]

    #the list order must be identical to that in class cNTLogFileReader, getDataLine() and getRawDataLine() !!!!
    #keeps information which is used for the various data formats
    def setToStandardNTLoggerItemList(self):
        self.clear()
        self.addItem( 'Time', 'ms', cDATATYPE_U64, cDATATYPE_FLOAT )
        self.addItem( 'Imu1rx', 'us', cDATATYPE_U8, cDATATYPE_Ux )
        self.addItem( 'Imu1done', 'us', cDATATYPE_U8, cDATATYPE_Ux )
        self.addItem( 'PIDdone', 'us', cDATATYPE_U8, cDATATYPE_Ux )
        self.addItem( 'Motdone', 'us', cDATATYPE_U8, cDATATYPE_Ux )
        self.addItem( 'Imu2rx', 'us', cDATATYPE_U8, cDATATYPE_Ux )
        self.addItem( 'Imu2done', 'us', cDATATYPE_U8, cDATATYPE_Ux )
        self.addItem( 'Logdone', 'us', cDATATYPE_U8, cDATATYPE_Ux )
        self.addItem( 'Loopdone', 'us', cDATATYPE_U8, cDATATYPE_Ux )

        self.addItem( 'State', 'uint', cDATATYPE_U16, cDATATYPE_Ux )
        self.addItem( 'Status', 'hex', cDATATYPE_U16, cDATATYPE_Ux )
        self.addItem( 'Status2', 'hex', cDATATYPE_U16, cDATATYPE_Ux )
        self.addItem( 'ErrorCnt', 'uint', cDATATYPE_U16, cDATATYPE_Ux )
        self.addItem( 'Voltage', 'V', cDATATYPE_U16, cDATATYPE_FLOAT )

        self.addItem( 'ax1', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'ay1', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'az1', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'gx1', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'gy1', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'gz1', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'Imu1State', 'hex', cDATATYPE_U8, cDATATYPE_Ux )

        self.addItem( 'ax2', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'ay2', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'az2', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'gx2', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'gy2', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'gz2', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'Imu2State', 'hex', cDATATYPE_U8, cDATATYPE_Ux )

        self.addItem( 'Imu1Pitch', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'Imu1Roll', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'Imu1Yaw', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'Imu2Pitch', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'Imu2Roll', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'Imu2Yaw', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )

        self.addItem( 'EncPitch', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'EncRoll', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'EncYaw', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'MotState', 'hex', cDATATYPE_U8, cDATATYPE_Ux )
        
        self.addItem( 'PIDPitch', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'PIDRoll', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'PIDYaw', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'PIDMotPitch', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'PIDMotRoll', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'PIDMotYaw', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )

        self.addItem( 'Rx1', 'g', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'Ry1', 'g', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'Rz1', 'g', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'AccAmp1', 'g', cDATATYPE_Ux, cDATATYPE_FLOAT ) #injected value
        self.addItem( 'AccConf1', 'uint', cDATATYPE_U16, cDATATYPE_FLOAT )
        self.addItem( 'YawTarget2', 'deg', cDATATYPE_S16, cDATATYPE_FLOAT )

        self.addItem( 'MotFlags', 'hex', cDATATYPE_U8, cDATATYPE_Ux )
        self.addItem( 'VmaxPitch', 'uint', cDATATYPE_U8, cDATATYPE_Ux )
        self.addItem( 'MotPitch', 'uint', cDATATYPE_U16, cDATATYPE_Ux )
        self.addItem( 'VmaxRoll', 'uint', cDATATYPE_U8, cDATATYPE_Ux )
        self.addItem( 'MotRoll', 'uint', cDATATYPE_U16, cDATATYPE_Ux )
        self.addItem( 'VmaxYaw', 'uint', cDATATYPE_U8, cDATATYPE_Ux )
        self.addItem( 'MotYaw', 'uint', cDATATYPE_U16, cDATATYPE_Ux )

        self.addItem( 'dUPitch', '', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'dURoll', '', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'dUYaw', '', cDATATYPE_S16, cDATATYPE_FLOAT )
#STL
        self.addItem( 'q0', '', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'q1', '', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'q2', '', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'q3', '', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'vx', 'm/s', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'vy', 'm/s', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'vz', 'm/s', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'YawRateCmd', 'int', cDATATYPE_S16, cDATATYPE_FLOAT )
        self.addItem( 'FCStatus', 'uint', cDATATYPE_U8, cDATATYPE_Ux )
        self.addItem( 'SLStatus', 'uint', cDATATYPE_U8, cDATATYPE_Ux )
        
        self.addItem( 'ax1raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'ay1raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'az1raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'gx1raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'gy1raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'gz1raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'T1', 'o', cDATATYPE_S16, cDATATYPE_FLOAT )

        self.addItem( 'ax2raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'ay2raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'az2raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'gx2raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'gy2raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'gz2raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'T2', 'o', cDATATYPE_S16, cDATATYPE_FLOAT )

        self.addItem( 'ax3raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'ay3raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'az3raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'gx3raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'gy3raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'gz3raw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'T3', 'o', cDATATYPE_S16, cDATATYPE_FLOAT )

        self.addItem( 'EncRawPitch', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'EncRawRoll', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'EncRawYaw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'dURawPitch', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'dURawRoll', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'dURawYaw', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        
        self.addItem( 'CameraCmd', 'int', cDATATYPE_U8, cDATATYPE_Ux )
        self.addItem( 'CameraPwm', 'int', cDATATYPE_U16, cDATATYPE_Ux )
#STL
        self.addItem( 'debug1', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'debug2', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'debug3', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'debug4', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'debug5', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'debug6', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'debug7', 'int', cDATATYPE_S16, cDATATYPE_Sx )
        self.addItem( 'inj1', '', cDATATYPE_Sx, cDATATYPE_FLOAT ) #injected value
        self.addItem( 'inj2', '', cDATATYPE_Sx, cDATATYPE_FLOAT ) #injected value
        self.addItem( 'inj3', '', cDATATYPE_Sx, cDATATYPE_FLOAT ) #injected value

    #self.graphSelectorList may be a larger list than that generated by getGraphSelectorList() 
    def getGraphSelectorDefaultIndex(self, graphSelectorList=None): #allows to check in a modified list
        if not graphSelectorList: graphSelectorList = self.graphSelectorList
        for i in range(len(graphSelectorList)):
            if graphSelectorList[i][0] == 'Imu1 Pitch,Roll,Yaw': return i
        for i in range(len(graphSelectorList)):
            if graphSelectorList[i][0] == 'Acc1': return i
        return None
        

###############################################################################
# cNTLogDataFrame
# this is the class to handle and host the data of one frame
#-----------------------------------------------------------------------------#
# NTLogger decodes the data on the NT bus and stores that decoded data on the SD card
# error handling:
#   there are two types of error, (i) a package is incomplete, (ii) a crucial package is not complete
#   self.error is set, depending on the general error type
#   each doXXX returns True or False, so that a parser can determine more detailed error conditions

cNTDATAFRAME_OK = 0
cNTDATAFRAME_CMDERROR = 1
cNTDATAFRAME_SETMOTERROR = 2
cNTDATAFRAME_SETLOGERROR = 4

cNTBUS_MOTOR_FLAG_FOC = 0x20 #is needed to distinguish "non-encoder" and  "encoder" versions of SetMotAll

#this is a base class
# it holds all data, and provides convenience functions to set them, and extract them as dataline and rawline
class cNTDataFrameObject:

    def __init__(self):
        self.logVersion = cLOGVERSION_NT_LATEST #allows to detect different log file versions, latest version as default
        self.Time = 0 #that's the actual time of a data frame

        self.clear()
                 
        #injected values
        self.Time = self.TimeStamp32
        self.fAhrs1AccAmp = 0.0
        self.EncAnglePitch,self.EncAngleRoll,self.EncAngleYaw = 0,0,0 #this is converted to deg
        self.dUPitch,self.dURoll,self.dUYaw = 0,0,0 #this is converted to 1
        self.dbg_inj1 = self.dbg_inj2 = self.dbg_inj3 = 0
#XX
        self.t_last = 0
        self.vn_x = self.vn_y = self.vn_z = 0
        self.an_x = self.an_y = self.an_z = 0
        
        self.vx_last = self.vy_last = self.vz_last = 0
        
    def setLogVersion(self,ver):
        self.logVersion = ver

    def getLogVersion(self):
        return self.logVersion

    def clear(self):
        self.TimeStamp32 = 0
        self.Imu1received,self.Imu1done,self.PIDdone,self.Motorsdone  = 0,0,0,0
        self.Imu2received,self.Imu2done,self.Logdone,self.Loopdone = 0,0,0,0
        self.State,self.Status,self.Status2,self.ErrorCnt,self.Voltage = 0,0,0,0,0
        self.Imu1AnglePitch,self.Imu1AngleRoll,self.Imu1AngleYaw = 0,0,0
        self.Imu2AnglePitch,self.Imu2AngleRoll,self.Imu2AngleYaw = 0,0,0
        self.EncRawPitch,self.EncRawRoll,self.EncRawYaw,self.MotState = 0,0,0,0 #this is the received, raw data
        
        self.Flags,self.VmaxPitch,self.MotPitch,self.VmaxRoll,self.MotRoll,self.VmaxYaw,self.MotYaw = 0,0,0,0,0,0,0
        self.dURawPitch,self.dURawRoll,self.dURawYaw = 0,0,0  #this store the received, raw data
#STL
        self.q0,self.q1,self.q2,self.q3 = 0,0,0,0
        self.vx,self.vy,self.vz = 0,0,0
        self.YawRateCmd,self.SLFCStatus,self.SLStatus = 0,0,0

        self.ax1,self.ay1,self.az1,self.gx1,self.gy1,self.gz1,self.Imu1State = 0,0,0,0,0,0,0
        self.ax2,self.ay2,self.az2,self.gx2,self.gy2,self.gz2,self.Imu2State = 0,0,0,0,0,0,0

        self.ax1raw,self.ay1raw,self.az1raw,self.gx1raw,self.gy1raw,self.gz1raw,self.temp1 = 0,0,0,0,0,0,0
        self.ax2raw,self.ay2raw,self.az2raw,self.gx2raw,self.gy2raw,self.gz2raw,self.temp2 = 0,0,0,0,0,0,0
        self.ax3raw,self.ay3raw,self.az3raw,self.gx3raw,self.gy3raw,self.gz3raw,self.temp3 = 0,0,0,0,0,0,0

        self.PIDCntrlPitch,self.PIDCntrlRoll,self.PIDCntrlYaw = 0,0,0
        self.PIDMotorCntrlPitch,self.PIDMotorCntrlRoll,self.PIDMotorCntrlYaw = 0,0,0

        self.Ahrs1Rx,self.Ahrs1Ry,self.Ahrs1Rz,self.Ahrs1AccConfidence,self.Ahrs1YawTarget = 0,0,0,0,0
        self.Ahrs2Rx,self.Ahrs2Ry,self.Ahrs2Rz,self.Ahrs2AccConfidence,self.Ahrs2YawTarget = 0,0,0,0,0

        self.CameraFlags,self.CameraModel,self.CameraCmd,self.CameraUnused,self.CameraPwm = 0,0,0,0,0
#STL
        self.debug1,self.debug2,self.debug3,self.debug4,self.debug5,self.debug6,self.debug7 = 0,0,0,0,0,0,0

        self.error = cNTDATAFRAME_OK #new frame, new game

    #some default functions to set values
    def setLogger_V0(self,tupel):
        (self.TimeStamp32,
         self.Imu1received,self.Imu1done,self.PIDdone,self.Motorsdone,
         self.Imu2received,self.Imu2done,self.Loopdone,
         self.State,self.Status,self.Status2,self.ErrorCnt,self.Voltage,
         self.Imu1AnglePitch,self.Imu1AngleRoll,self.Imu1AngleYaw,
         self.Imu2AnglePitch,self.Imu2AngleRoll,self.Imu2AngleYaw,
        ) = tupel
    def setLogger_V3(self,tupel):
        (self.TimeStamp32,
         self.Imu1received,self.Imu1done,self.PIDdone,self.Motorsdone,
         self.Imu2done,self.Logdone,self.Loopdone,
         self.State,self.Status,self.Status2,self.ErrorCnt,self.Voltage,
         self.Imu1AnglePitch,self.Imu1AngleRoll,self.Imu1AngleYaw,
         self.Imu2AnglePitch,self.Imu2AngleRoll,self.Imu2AngleYaw,
        ) = tupel
        
    def setMotorAll(self,tupel):
        (self.Flags,self.VmaxPitch,self.MotPitch,self.VmaxRoll,self.MotRoll,self.VmaxYaw,self.MotYaw) = tupel
        self.dURawPitch,self.dURawRoll,self.dURawYaw = 0,0,0

    def setMotorAllVFoc(self,tupel):
        (self.Flags,self.dURawPitch,self.dURawRoll,self.dURawYaw) = tupel
        self.VmaxPitch,self.MotPitch,self.VmaxRoll,self.MotRoll,self.VmaxYaw,self.MotYaw = 0,0,0,0,0,0
        
    def setCamera(self,tupel):
        (self.CameraFlags,self.CameraModel,self.CameraCmd,self.CameraUnused,self.CameraPwm) = tupel

    def cmdAccGyro1_V2(self,tupel):
        (self.ax1,self.ay1,self.az1,self.gx1,self.gy1,self.gz1,self.Imu1State) = tupel

    def cmdAccGyro2_V2(self,tupel):
        (self.ax2,self.ay2,self.az2,self.gx2,self.gy2,self.gz2,self.Imu2State) = tupel

    def cmdAccGyro1Raw_V2(self,tupel):
        (self.ax1raw,self.ay1raw,self.az1raw,self.gx1raw,self.gy1raw,self.gz1raw,self.temp1) = tupel

    def cmdAccGyro2Raw_V2(self,tupel):
        (self.ax2raw,self.ay2raw,self.az2raw,self.gx2raw,self.gy2raw,self.gz2raw,self.temp2) = tupel

    def cmdAccGyro3Raw_V2(self,tupel):
        (self.ax3raw,self.ay3raw,self.az3raw,self.gx3raw,self.gy3raw,self.gz3raw,self.temp3) = tupel

    def cmdEncoder(self,tupel):
        (self.EncRawPitch,self.EncRawRoll,self.EncRawYaw,self.MotState) = tupel
        
    def cmdPid(self,tupel):
        (self.PIDCntrlPitch,self.PIDCntrlRoll,self.PIDCntrlYaw,
         self.PIDMotorCntrlPitch,self.PIDMotorCntrlRoll,self.PIDMotorCntrlYaw) = tupel

    def cmdAhrs1(self,tupel):
        (self.Ahrs1Rx,self.Ahrs1Ry,self.Ahrs1Rz,self.Ahrs1AccConfidence,self.Ahrs1YawTarget) = tupel

    def cmdAhrs2(self,tupel):
        (self.Ahrs2Rx,self.Ahrs2Ry,self.Ahrs2Rz,self.Ahrs2AccConfidence,self.Ahrs2YawTarget) = tupel

    def cmdAccGyro_V1(self,tupel):
        (self.ax1,self.ay1,self.az1,self.gx1,self.gy1,self.gz1,self.temp1,self.Imu1State,
         self.ax2,self.ay2,self.az2,self.gx2,self.gy2,self.gz2,self.temp2,self.Imu2State) = tupel

    def cmdAccGyro1Raw_V1(self,tupel):
        (self.ax1raw,self.ay1raw,self.az1raw,self.gx1raw,self.gy1raw,self.gz1raw) = tupel

    def cmdAccGyro2Raw_V1(self,tupel):
        (self.ax2raw,self.ay2raw,self.az2raw,self.gx2raw,self.gy2raw,self.gz2raw) = tupel

    def cmdAccGyro3Raw_V1(self,tupel):
        (self.ax3raw,self.ay3raw,self.az3raw,self.gx3raw,self.gy3raw,self.gz3raw) = tupel

    def cmdParameter(self,tupel):
        (self.ParameterAdr,self.ParameterValue,self.ParameterFormat,self.ParameterNameStr) = tupel
        if self.ParameterFormat==4: #MAV_PARAM_TYPE_INT16 = 4
            if self.ParameterValue>32768: self.ParameterValue -= 65536
#STL            
    def cmdStorM32LinkData(self,tupel):
        (self.q0,self.q1,self.q2,self.q3,self.vx,self.vy,self.vz,
         self.YawRateCmd,self.SLFCStatus,self.SLStatus) = tupel
    def cmdDebugData(self,tupel):
        (self.debug1,self.debug2,self.debug3,self.debug4,self.debug5,self.debug6,self.debug7) = tupel

    #some default prototypes, are all called by parser
    def doSetLogger(self,payload): return True
    def doSetMotorAll(self,payload): return True
    def doSetMotorAllVFoc(self,payload): return True
    def doSetCamera(self,payload): return True
    def doCmdAccGyro1_V2(self,payload): return True
    def doCmdAccGyro2_V2(self,payload): return True
    def doCmdAccGyro1Raw_V2(self,payload): return True
    def doCmdAccGyro2Raw_V2(self,payload): return True
    def doCmdAccGyro3Raw_V2(self,payload): return True
    def doCmdEncoder(self,payload): return True
    def doCmdPid(self,payload): return True
    def doCmdAhrs1(self,payload): return True
    def doCmdAhrs2(self,payload): return True
    def doCmdAccGyro_V1(self,payload): return True
    def doCmdAccGyro1Raw_V1(self,payload): return True
    def doCmdAccGyro2Raw_V1(self,payload): return True
    def doCmdAccGyro3Raw_V1(self,payload): return True
    def doCmdParameter(self,payload): return True
#STL            
    def doCmdStorM32LinkData(self,payload): return True
    def doCmdDebugData(self,payload): return True

    def readCmdByte(self): return 255

    def calculateTime(self,datalog_TimeStamp32_start):
        self.Time = self.TimeStamp32 - datalog_TimeStamp32_start

    def calculateInjectedValues(self):
        self.fAhrs1AccAmp = sqrt(self.ax1*self.ax1 + self.ay1*self.ay1 + self.az1*self.az1)*10000.0/8192.0

        self.EncAnglePitch = self.EncRawPitch * 360.0 / 65536.0 #raw is int16
        self.EncAngleRoll = self.EncRawRoll * 360.0 / 65536.0 #raw is int16
        self.EncAngleYaw = self.EncRawYaw * 360.0 / 65536.0 #raw is int16
        self.dUPitch = self.dURawPitch / 65536.0 #raw is q16
        self.dURoll = self.dURawRoll / 65536.0 #raw is q16
        self.dUYaw = self.dURawYaw / 65536.0 #raw is q16
       
    #allows to do tests to check the integrety of teh data inthe frame, and to skip if invalid
    def isValid(self):
        if self.State>100: return False
        return True 
        
    #------------------------------------------
    #NTbus data logs: the order must match that of setToStandardNTLoggerItemList()
    def getDataLine(self):
        dataline = ''
        dataline +=  '{:.1f}'.format(0.001*self.Time) + "\t"

        dataline +=  str(10*self.Imu1received) + "\t"
        dataline +=  str(10*self.Imu1done) + "\t"
        dataline +=  str(10*self.PIDdone) + "\t"
        dataline +=  str(10*self.Motorsdone) + "\t"
        dataline +=  str(10*self.Imu2received) + "\t"
        dataline +=  str(10*self.Imu2done) + "\t"
        dataline +=  str(10*self.Logdone) + "\t"
        dataline +=  str(10*self.Loopdone) + "\t"

        dataline +=  str(self.State) + "\t"
        dataline +=  str(self.Status) + "\t"
        dataline +=  str(self.Status2) + "\t"
        dataline +=  str(self.ErrorCnt) + "\t"
        dataline +=  '{:.3f}'.format(0.001 * self.Voltage) + "\t"

        dataline +=  str(self.ax1) + "\t" + str(self.ay1) + "\t" + str(self.az1) + "\t"
        dataline +=  str(self.gx1) + "\t" + str(self.gy1) + "\t" + str(self.gz1) + "\t"
        dataline +=  str(self.Imu1State) + "\t"
        dataline +=  str(self.ax2) + "\t" + str(self.ay2) + "\t" + str(self.az2) + "\t"
        dataline +=  str(self.gx2) + "\t" + str(self.gy2) + "\t" + str(self.gz2) + "\t"
        dataline +=  str(self.Imu2State) + "\t"

        if self.logVersion==cLOGVERSION_NT_V3:
            dataline +=  '{:.3f}'.format( 0.001 * self.Imu1AnglePitch ) + "\t"
            dataline +=  '{:.3f}'.format( 0.001 * self.Imu1AngleRoll ) + "\t"
            dataline +=  '{:.3f}'.format( 0.001 * self.Imu1AngleYaw ) + "\t"
            dataline +=  '{:.3f}'.format( 0.001 * self.Imu2AnglePitch ) + "\t"
            dataline +=  '{:.3f}'.format( 0.001 * self.Imu2AngleRoll ) + "\t"
            dataline +=  '{:.3f}'.format( 0.001 * self.Imu2AngleYaw ) + "\t"
        else:
            dataline +=  '{:.2f}'.format( 0.01 * self.Imu1AnglePitch ) + "\t"
            dataline +=  '{:.2f}'.format( 0.01 * self.Imu1AngleRoll ) + "\t"
            dataline +=  '{:.2f}'.format( 0.01 * self.Imu1AngleYaw ) + "\t"
            dataline +=  '{:.2f}'.format( 0.01 * self.Imu2AnglePitch ) + "\t"
            dataline +=  '{:.2f}'.format( 0.01 * self.Imu2AngleRoll ) + "\t"
            dataline +=  '{:.2f}'.format( 0.01 * self.Imu2AngleYaw ) + "\t"

        dataline +=  '{:.3f}'.format( self.EncAnglePitch ) + "\t" ##XX Check that!
        dataline +=  '{:.3f}'.format( self.EncAngleRoll ) + "\t"
        dataline +=  '{:.3f}'.format( self.EncAngleYaw ) + "\t"
        dataline +=  str(self.MotState) + "\t"

        dataline +=  '{:.2f}'.format( 0.01 * self.PIDCntrlPitch ) + "\t"
        dataline +=  '{:.2f}'.format( 0.01 * self.PIDCntrlRoll ) + "\t"
        dataline +=  '{:.2f}'.format( 0.01 * self.PIDCntrlYaw ) + "\t"
        dataline +=  '{:.2f}'.format( 0.01 * self.PIDMotorCntrlPitch ) + "\t"
        dataline +=  '{:.2f}'.format( 0.01 * self.PIDMotorCntrlRoll ) + "\t"
        dataline +=  '{:.2f}'.format( 0.01 * self.PIDMotorCntrlYaw ) + "\t"

        dataline +=  '{:.4f}'.format(0.0001 * self.Ahrs1Rx) + "\t"
        dataline +=  '{:.4f}'.format(0.0001 * self.Ahrs1Ry) + "\t"
        dataline +=  '{:.4f}'.format(0.0001 * self.Ahrs1Rz) + "\t"
        dataline +=  '{:.4f}'.format(0.0001 * self.fAhrs1AccAmp) + "\t"
        dataline +=  '{:.4f}'.format(0.0001 * self.Ahrs1AccConfidence) + "\t"
        dataline +=  '{:.2f}'.format(0.01 * self.Ahrs1YawTarget) + "\t"
        
        dataline +=  str(self.Flags) + "\t"
        dataline +=  str(self.VmaxPitch) + "\t" + str(self.MotPitch) + "\t"
        dataline +=  str(self.VmaxRoll) + "\t"  + str(self.MotRoll) + "\t"
        dataline +=  str(self.VmaxYaw) + "\t"   + str(self.MotYaw) + "\t"
        dataline +=  str(self.dUPitch) + "\t" + str(self.dURoll) + "\t" + str(self.dUYaw) + "\t"
#STL
        dataline +=  '{:.4f}'.format(0.0001 * self.q0) + "\t"
        dataline +=  '{:.4f}'.format(0.0001 * self.q1) + "\t"
        dataline +=  '{:.4f}'.format(0.0001 * self.q2) + "\t"
        dataline +=  '{:.4f}'.format(0.0001 * self.q3) + "\t"
        dataline +=  '{:.2f}'.format(0.01 * self.vx) + "\t"
        dataline +=  '{:.2f}'.format(0.01 * self.vy) + "\t"
        dataline +=  '{:.2f}'.format(0.01 * self.vz) + "\t"
        dataline +=  str(self.YawRateCmd) + "\t"
        dataline +=  str(self.SLFCStatus) + "\t" + str(self.SLStatus) + "\t"

        dataline +=  str(self.ax1raw) + "\t" + str(self.ay1raw) + "\t" + str(self.az1raw) + "\t"
        dataline +=  str(self.gx1raw) + "\t" + str(self.gy1raw) + "\t" + str(self.gz1raw) + "\t"
        dataline +=  '{:.2f}'.format(0.01 * self.temp1) + "\t"

        dataline +=  str(self.ax2raw) + "\t" + str(self.ay2raw) + "\t" + str(self.az2raw) + "\t"
        dataline +=  str(self.gx2raw) + "\t" + str(self.gy2raw) + "\t" + str(self.gz2raw) + "\t"
        dataline +=  '{:.2f}'.format(0.01 * self.temp2) + "\t"

        dataline +=  str(self.ax3raw) + "\t" + str(self.ay3raw) + "\t" + str(self.az3raw) + "\t"
        dataline +=  str(self.gx3raw) + "\t" + str(self.gy3raw) + "\t" + str(self.gz3raw) + "\t"
        dataline +=  '{:.2f}'.format(0.01 * self.temp3) + "\t"

        dataline +=  str( self.EncRawPitch ) + "\t" + str( self.EncRawRoll ) + "\t" + str( self.EncRawYaw ) + "\t"
        dataline +=  str(self.dURawPitch) + "\t" + str(self.dURawRoll) + "\t" + str(self.dURawYaw) + "\t"
        
        dataline +=  str(self.CameraCmd) + "\t" + str(self.CameraPwm) + "\t"

#STL
#XX        dataline +=  str(self.debug1) + "\t" +  str(self.debug2) + "\t" +  str(self.debug3) + "\t"
#XX        dataline +=  str(self.debug4) + "\t" +  str(self.debug5) + "\t" +  str(self.debug6) + "\t"
#XX        dataline +=  str(self.dbg_inj1) + "\t" +  str(self.dbg_inj2) + "\t" +  str(self.dbg_inj3) + "\t"
#angles, velocities
#        sform = '{:.2f}'; sfact = 0.01
#accelerations
#        sform = '{:.3f}'; sfact = 0.001
        sform = '{:.3f}'; sfact = 0.001
        dataline +=  sform.format(sfact*self.debug1) + "\t" +  sform.format(sfact*self.debug2) + "\t" +  sform.format(sfact*self.debug3) + "\t"
        dataline +=  sform.format(sfact*self.debug4) + "\t" +  sform.format(sfact*self.debug5) + "\t" +  sform.format(sfact*self.debug6) + "\t"
##        dataline +=  str(self.ax1*9.81/8048) + "\t" + str(self.ay1*9.81/8048) + "\t" + str(self.az1*9.81/8048) + "\t"

        dataline +=  str(self.debug7) + "\t" # = numSat
#        accest = sqrt(self.debug1*self.debug1 + self.debug2*self.debug2)
#        dataline +=  sform.format(sfact*accest) + "\t"
        
        self.dbg_inj1 = self.ax1*9.81/8048 - sfact*self.debug4
        self.dbg_inj2 = self.ay1*9.81/8048 - sfact*self.debug5
        self.dbg_inj3 = self.az1*9.81/8048 - sfact*self.debug6
        dataline +=  str(self.dbg_inj1) + "\t" + str(self.dbg_inj2) + "\t" + str(self.dbg_inj3) + "\n"
        
#velocity
#        sform = '{:.2f}'; sfact = 0.01
#        dataline +=  sform.format(sfact*self.stl_debug4) + "\t" +  sform.format(sfact*self.stl_debug5) + "\t" +  sform.format(sfact*self.stl_debug6) + "\t"

        alpha = 0.4
        beta = 0.2
        
        if self.SLStatus >= 125: #new value
        
            dt = 0.000001*self.Time - self.t_last
            self.t_last = 0.000001*self.Time
            
            sform = '{:.2f}'; sfact = 0.01
            vx_meas = sfact*self.vx
            vy_meas = sfact*self.vy
            vz_meas = sfact*self.vz
            
            vx_pred = self.vn_x + self.an_x * dt
            vy_pred = self.vn_y + self.an_y * dt
            vz_pred = self.vn_z + self.an_z * dt
        
            vx_innov = vx_meas - vx_pred
            vy_innov = vy_meas - vy_pred
            vz_innov = vz_meas - vz_pred
        
            self.vn_x = vx_pred + alpha * vx_innov  # (1-alpha) v_x + alpha debug4
            self.vn_y = vy_pred + alpha * vy_innov
            self.vn_z = vz_pred + alpha * vz_innov
        
            if self.t_last > 0.0000000001:
                self.an_x += beta/dt * vx_innov
                self.an_y += beta/dt * vy_innov
                self.an_z += beta/dt * vz_innov
                
                self.dbg_inj1 = (vx_meas - self.vx_last )/dt
                self.dbg_inj2 = (vy_meas - self.vy_last )/dt
                self.dbg_inj3 = (vz_meas - self.vz_last )/dt

            #just to keep them 
            self.vx_last = vx_meas
            self.vy_last = vy_meas
            self.vz_last = vz_meas
       
#        dataline +=  str(self.an_x) + "\t" +  str(self.vn_x) + "\t" +  str(self.stl_inj1) + "\t" #injected
      
        
        return dataline

    #------------------------------------------
    #NTbus data logs: the order must match that of setToStandardNTLoggerItemList()
    def getRawDataLine(self):
        rawdataline = []
        rawdataline.append(self.Time)

        rawdataline.append(10*self.Imu1received)
        rawdataline.append(10*self.Imu1done)
        rawdataline.append(10*self.PIDdone)
        rawdataline.append(10*self.Motorsdone)
        rawdataline.append(10*self.Imu2received)
        rawdataline.append(10*self.Imu2done)
        rawdataline.append(10*self.Logdone)
        rawdataline.append(10*self.Loopdone)

        rawdataline.append(self.State)
        rawdataline.append(self.Status)
        rawdataline.append(self.Status2)
        rawdataline.append(self.ErrorCnt)
        rawdataline.append(int(self.Voltage))

        rawdataline.append(self.ax1)
        rawdataline.append(self.ay1)
        rawdataline.append(self.az1)
        rawdataline.append(self.gx1)
        rawdataline.append(self.gy1)
        rawdataline.append(self.gz1)
        rawdataline.append(self.Imu1State)
        rawdataline.append(self.ax2)
        rawdataline.append(self.ay2)
        rawdataline.append(self.az2)
        rawdataline.append(self.gx2)
        rawdataline.append(self.gy2)
        rawdataline.append(self.gz2)
        rawdataline.append(self.Imu2State)
        
        rawdataline.append(self.Imu1AnglePitch)
        rawdataline.append(self.Imu1AngleRoll)
        rawdataline.append(self.Imu1AngleYaw)
        rawdataline.append(self.Imu2AnglePitch)
        rawdataline.append(self.Imu2AngleRoll)
        rawdataline.append(self.Imu2AngleYaw)

        rawdataline.append(self.EncAnglePitch)
        rawdataline.append(self.EncAngleRoll)
        rawdataline.append(self.EncAngleYaw)
        rawdataline.append(self.MotState)

        rawdataline.append(self.PIDCntrlPitch)
        rawdataline.append(self.PIDCntrlRoll)
        rawdataline.append(self.PIDCntrlYaw)
        rawdataline.append(self.PIDMotorCntrlPitch)
        rawdataline.append(self.PIDMotorCntrlRoll)
        rawdataline.append(self.PIDMotorCntrlYaw)

        rawdataline.append(self.Ahrs1Rx)
        rawdataline.append(self.Ahrs1Ry)
        rawdataline.append(self.Ahrs1Rz)
        rawdataline.append(int(self.fAhrs1AccAmp))
        rawdataline.append(self.Ahrs1AccConfidence)
        rawdataline.append(self.Ahrs1YawTarget)

        rawdataline.append(self.Flags)
        rawdataline.append(self.VmaxPitch)
        rawdataline.append(self.MotPitch)
        rawdataline.append(self.VmaxRoll)
        rawdataline.append(self.MotRoll)
        rawdataline.append(self.VmaxYaw)
        rawdataline.append(self.MotYaw)

        rawdataline.append(self.dUPitch)
        rawdataline.append(self.dURoll)
        rawdataline.append(self.dUYaw)
#STL
        rawdataline.append(self.q0)
        rawdataline.append(self.q1)
        rawdataline.append(self.q2)
        rawdataline.append(self.q3)
        rawdataline.append(self.vx)
        rawdataline.append(self.vy)
        rawdataline.append(self.vz)
        rawdataline.append(self.YawRateCmd)
        rawdataline.append(self.SLFCStatus)
        rawdataline.append(self.SLStatus)

        rawdataline.append(self.ax1raw)
        rawdataline.append(self.ay1raw)
        rawdataline.append(self.az1raw)
        rawdataline.append(self.gx1raw)
        rawdataline.append(self.gy1raw)
        rawdataline.append(self.gz1raw)
        rawdataline.append(self.temp1)

        rawdataline.append(self.ax2raw)
        rawdataline.append(self.ay2raw)
        rawdataline.append(self.az2raw)
        rawdataline.append(self.gx2raw)
        rawdataline.append(self.gy2raw)
        rawdataline.append(self.gz2raw)
        rawdataline.append(self.temp2)

        rawdataline.append(self.ax3raw)
        rawdataline.append(self.ay3raw)
        rawdataline.append(self.az3raw)
        rawdataline.append(self.gx3raw)
        rawdataline.append(self.gy3raw)
        rawdataline.append(self.gz3raw)
        rawdataline.append(self.temp3)

        rawdataline.append(self.EncRawPitch)
        rawdataline.append(self.EncRawRoll)
        rawdataline.append(self.EncRawYaw)
        rawdataline.append(self.dURawPitch)
        rawdataline.append(self.dURawRoll)
        rawdataline.append(self.dURawYaw)
        
        rawdataline.append(self.CameraCmd)
        rawdataline.append(self.CameraPwm)
#STL        
        rawdataline.append(self.debug1)
        rawdataline.append(self.debug2)
        rawdataline.append(self.debug3)
        rawdataline.append(self.debug4)
        rawdataline.append(self.debug5)
        rawdataline.append(self.debug6)
        rawdataline.append(self.debug7)
        rawdataline.append(self.dbg_inj1)
        rawdataline.append(self.dbg_inj2)
        rawdataline.append(self.dbg_inj3)

        return rawdataline


#this is a child class for handling NT log files
# it does the unpacking of the received payload bytes 
class cNTLogFileDataFrame(cNTDataFrameObject):

    def __init__(self):
        super().__init__()

        #structures of data as stored in NT log files, recorded by a NT Logger
        self.setLoggerStruct_V0 = struct.Struct('=I'+'BBBBBBB'+'HHHHH'+'hhhhhh')
        self.setLoggerStruct_V3 = struct.Struct('=I'+'BBBBBBB'+'HHHHH'+'iiiiii')
        self.setMotorAllStruct = struct.Struct('=BBhBhBh')
        self.setCameraStruct = struct.Struct('=BBBBH')
        self.cmdAccGyroStruct_V1 = struct.Struct('=hhhhhhhB'+'hhhhhhhB')
        self.cmdAccGyroStruct_V2 = struct.Struct('=hhhhhhB')
        self.cmdAccGyroRawStruct_V1 = struct.Struct('=hhhhhh')
        self.cmdAccGyroRawStruct_V2 = struct.Struct('=hhhhhhh')
        self.cmdPidStruct = struct.Struct('=hhhhhh')
        self.cmdAhrsStruct = struct.Struct('=hhhhh')
        self.cmdEncoderStruct = struct.Struct('=hhhB')
        self.cmdParameterStruct = struct.Struct('=HHH16s')
#STL        
        self.cmdSTorM32LinkDataStruct = struct.Struct('=hhhhhhhhBB')
        self.cmdDebugDataStruct = struct.Struct('=hhhhhhh')

    def unpackSetLogger(self,payload):
        if self.logVersion==cLOGVERSION_NT_V3:
            self.setLogger_V3( self.setLoggerStruct_V3.unpack(payload) )
        else:
            self.setLogger_V0( self.setLoggerStruct_V0.unpack(payload) )

    def unpackSetMotorAll(self,payload): #struct.Struct('=BBhBhBh')
        (self.Flags,self.VmaxPitch,self.MotPitch,self.VmaxRoll,self.MotRoll,self.VmaxYaw,self.MotYaw
         ) = self.setMotorAllStruct.unpack(payload)
        #if( ntbus_buf[0] & NTBUS_MOTOR_FLAG_FOC ){
        if self.Flags&0x20 > 0:
            #dU = (u16)(ntbus_buf[1] & 0x7F) + ((u16)(ntbus_buf[2] & 0x7F ) << 7) + ((u16)(ntbus_buf[3] & 0x7F ) << 14); //low byte, high byte, highest byte
            #if( dU & (1<<20) ) dU |= 0xFFF00000; //if 20th bit is set, restore that it's a negative value
            #do this first, so that we have the original values
            # the brackets are most important!
            self.dURawPitch = (self.VmaxPitch&0x007f) +  ((self.MotPitch&0x007f) << 7) + ((self.MotPitch&0x7f00) << 6)
            if( self.dURawPitch > (1<<20) ): self.dURawPitch = self.dURawPitch - (1<<21)
            self.dURawRoll = (self.VmaxRoll&0x007f) +  ((self.MotRoll&0x007f) << 7) + ((self.MotRoll&0x7f00) << 6)
            if( self.dURawRoll > (1<<20) ): self.dURawRoll = self.dURawRoll - (1<<21)
            self.dURawYaw = (self.VmaxYaw&0x007f) +  ((self.MotYaw&0x007f) << 7) + ((self.MotYaw&0x7f00) << 6)
            if( self.dURawYaw > (1<<20) ): self.dURawYaw = self.dURawYaw - (1<<21)
            self.VmaxPitch = self.MotPitch = self.VmaxRoll = self.MotRoll = self.VmaxYaw = self.MotYaw = 0
        else:
            #Vmax,Mot were decoded alread by NTLogger
            self.dURawPitch = self.dURawRoll = self.dURawYaw = 0
            
    def unpackSetCamera(self,payload): #struct.Struct('=BBBH')
        (self.CameraFlags,self.CameraModel,self.CameraCmd,self.CameraUnused,self.CameraPwm
         ) = self.setCameraStruct.unpack(payload)

    def unpackCmdAccGyro1_V2(self,payload): #struct.Struct('=hhhhhhB')
        (self.ax1,self.ay1,self.az1,self.gx1,self.gy1,self.gz1,self.Imu1State
         ) = self.cmdAccGyroStruct_V2.unpack(payload)

    def unpackCmdAccGyro2_V2(self,payload): #struct.Struct('=hhhhhhB')
        (self.ax2,self.ay2,self.az2,self.gx2,self.gy2,self.gz2,self.Imu2State
         ) = self.cmdAccGyroStruct_V2.unpack(payload)

    def unpackCmdAccGyro1Raw_V2(self,payload): #struct.Struct('=hhhhhhh')
        (self.ax1raw,self.ay1raw,self.az1raw,self.gx1raw,self.gy1raw,self.gz1raw,self.temp1
         ) = self.cmdAccGyroRawStruct_V2.unpack(payload)

    def unpackCmdAccGyro2Raw_V2(self,payload): #struct.Struct('=hhhhhhh')
        (self.ax2raw,self.ay2raw,self.az2raw,self.gx2raw,self.gy2raw,self.gz2raw,self.temp2
         ) =  self.cmdAccGyroRawStruct_V2.unpack(payload)

    def unpackCmdAccGyro3Raw_V2(self,payload): #struct.Struct('=hhhhhhh')
        (self.ax3raw,self.ay3raw,self.az3raw,self.gx3raw,self.gy3raw,self.gz3raw,self.temp3
         ) = self.cmdAccGyroRawStruct_V2.unpack(payload)

    def unpackCmdPid(self,payload): #struct.Struct('=hhhhhh')
        (self.PIDCntrlPitch,self.PIDCntrlRoll,self.PIDCntrlYaw,
         self.PIDMotorCntrlPitch,self.PIDMotorCntrlRoll,self.PIDMotorCntrlYaw
         ) = self.cmdPidStruct.unpack(payload)

    def unpackCmdAhrs1(self,payload): #struct.Struct('=hhhhh')
        (self.Ahrs1Rx,self.Ahrs1Ry,self.Ahrs1Rz,self.Ahrs1AccConfidence,self.Ahrs1YawTarget
         ) = self.cmdAhrsStruct.unpack(payload)

    def unpackCmdAhrs2(self,payload): #struct.Struct('=hhhhh')
        (self.Ahrs2Rx,self.Ahrs2Ry,self.Ahrs2Rz,self.Ahrs2AccConfidence,self.Ahrs2YawTarget
         ) = self.cmdAhrsStruct.unpack(payload)

    def unpackCmdAccGyro_V1(self,payload): #struct.Struct('=hhhhhhhB'+'hhhhhhhB')
        (self.ax1,self.ay1,self.az1,self.gx1,self.gy1,self.gz1,self.temp1,self.Imu1State,
         self.ax2,self.ay2,self.az2,self.gx2,self.gy2,self.gz2,self.temp2,self.Imu2State
         ) = self.cmdAccGyroStruct_V1.unpack(payload)

    def unpackCmdAccGyro1Raw_V1(self,payload): #struct.Struct('=hhhhhh')
        (self.ax1raw,self.ay1raw,self.az1raw,self.gx1raw,self.gy1raw,self.gz1raw
         ) = self.cmdAccGyroRawStruct_V1.unpack(payload)

    def unpackCmdAccGyro2Raw_V1(self,payload):
        (self.ax2raw,self.ay2raw,self.az2raw,self.gx2raw,self.gy2raw,self.gz2raw
         ) = self.cmdAccGyroRawStruct_V1.unpack(payload)

    def unpackCmdAccGyro3Raw_V1(self,payload):
        (self.ax3raw,self.ay3raw,self.az3raw,self.gx3raw,self.gy3raw,self.gz3raw
         ) = self.cmdAccGyroRawStruct_V1.unpack(payload)

    def unpackCmdEncoder(self,payload): #struct.Struct('=hhhB')
        (self.EncRawPitch,self.EncRawRoll,self.EncRawYaw,self.MotState
         ) = self.cmdEncoderStruct.unpack(payload)

    def unpackCmdParameter(self,payload):
        (self.ParameterAdr,self.ParameterValue,self.ParameterFormat,self.ParameterNameStr
         ) = self.cmdParameterStruct.unpack(payload)
        if self.ParameterFormat==4: #MAV_PARAM_TYPE_INT16 = 4
            if self.ParameterValue>32768: self.ParameterValue -= 65536
#STL            
    def unpackCmdSTorM32LinkData(self,payload): #struct.Struct('=hhhhhhhhBB')
        (self.q0,self.q1,self.q2,self.q3,self.vx,self.vy,self.vz,
        self.YawRateCmd,self.SLFCStatus,self.SLStatus
         ) = self.cmdSTorM32LinkDataStruct.unpack(payload)
    def unpackCmdDebugData(self,payload): #struct.Struct('=hhhhhhh')
        (self.debug1,self.debug2,self.debug3,self.debug4,self.debug5,self.debug6,self.debug7,
         ) = self.cmdDebugDataStruct.unpack(payload)

    def doSetLogger(self,payload): self.unpackSetLogger(payload); return True
    def doSetMotorAll(self,payload): self.unpackSetMotorAll(payload); return True
    def doSetCamera(self,payload): self.unpackSetCamera(payload); return True
    def doCmdAccGyro1_V2(self,payload): self.unpackCmdAccGyro1_V2(payload); return True
    def doCmdAccGyro2_V2(self,payload): self.unpackCmdAccGyro2_V2(payload); return True
    def doCmdAccGyro1Raw_V2(self,payload): self.unpackCmdAccGyro1Raw_V2(payload); return True
    def doCmdAccGyro2Raw_V2(self,payload): self.unpackCmdAccGyro2Raw_V2(payload); return True
    def doCmdAccGyro3Raw_V2(self,payload): self.unpackCmdAccGyro3Raw_V2(payload); return True
    def doCmdPid(self,payload): self.unpackCmdPid(payload); return True
    def doCmdAhrs1(self,payload): self.unpackCmdAhrs1(payload); return True
    def doCmdAhrs2(self,payload): self.unpackCmdAhrs2(payload); return True
    def doCmdAccGyro_V1(self,payload): self.unpackCmdAccGyro_V1(payload); return True
    def doCmdAccGyro1Raw_V1(self,payload): self.unpackCmdAccGyro1Raw_V1(payload); return True
    def doCmdAccGyro2Raw_V1(self,payload): self.unpackCmdAccGyro2Raw_V1(payload); return True
    def doCmdAccGyro3Raw_V1(self,payload): self.unpackCmdAccGyro3Raw_V1(payload); return True
    def doCmdEncoder(self,payload): self.unpackCmdEncoder(payload); return True
    def doCmdParameter(self,payload): self.unpackCmdParameter(payload); return True
#STL
    def doCmdSTorM32LinkData(self,payload): self.unpackCmdSTorM32LinkData(payload); return True
    def doCmdDebugData(self,payload): self.unpackCmdDebugData(payload); return True

    
cSETLOGGER_V3_DATALEN             = 36
cSETLOGGER_V3_HIGHBITSLEN         = 6
cSETLOGGER_V3_FRAMELEN            = 36 + 6 #+ 1
cSETMOTORALL_DATALEN              = 10
cSETMOTORALL_FRAMELEN             = 10 #10 + 1
cSETCAMERA_DATALEN                = 5
cSETCAMERA_FRAMELEN               = 5 #5 + 1
cCMDENCODERDATA_DATALEN           = 7
cCMDENCODERDATA_HIGHBITSLEN       = 2
cCMDENCODERDATA_FRAMELEN          = 7 + 2 #7 + 2 + 1
cCMDACCGYRODATA_V2_DATALEN        = 13
cCMDACCGYRODATA_V2_HIGHBITSLEN    = 2
cCMDACCGYRODATA_V2_FRAMELEN       = 13 + 2 #13 + 2 + 1
cCMDACCGYRORAWDATA_V2_DATALEN     = 14
cCMDACCGYRORAWDATA_V2_HIGHBITSLEN = 2
cCMDACCGYRORAWDATA_V2_FRAMELEN    = 14 + 2 #14 + 2 + 1
cCMDPIDDATA_DATALEN               = 12
cCMDPIDDATA_HIGHBITSLEN           = 2
cCMDPIDDATA_FRAMELEN              = 12 + 2 #12 + 2 + 1
cCMDAHRSDATA_DATALEN              = 10
cCMDAHRSDATA_HIGHBITSLEN          = 2
cCMDAHRSDATA_FRAMELEN             = 10 + 2 #10 + 2 + 1
#STL
cCMDSTORM32LINKDATA_DATALEN       = 18
cCMDSTORM32LINKDATA_HIGHBITSLEN   = 4
cCMDSTORM32LINKDATA_FRAMELEN      = 18 + 4 #18 + 2 + 1
cCMDDEBUGDATA_DATALEN             = 14
cCMDDEBUGDATA_HIGHBITSLEN         = 2
cCMDDEBUGDATA_FRAMELEN            = 14 + 2 #18 + 2 + 1

#this is a child class for handling NT serial data streams
# it does the encoding of the data onto the NT bus
#the reader must provide a function
# reader.readPayload(length)
#
# a frame error must ONLY be thrown, then one of the crucial packages is wrong,
# i.e. SetLogger, SetMotorAll !!
class cNTSerialDataFrame(cNTLogFileDataFrame):

    def __init__(self, _reader):
        super().__init__()
        self.reader = _reader

        #structures of data as transmitted on the NT bus, recorded by a USB-TTL adapter
        # differs for SetMotAll, SetLog, and SetCamera from  the data in a NT log file
        # these thus need special handling
        self.setLoggerStruct_V3_NTbus = struct.Struct('=I'+'BBBBBBB'+'HHHHH'+'hhhhhh'+'BBB')
        self.setCameraStruct_NTBus = struct.Struct('=BBBBB')

        self.setLogVersion(cLOGVERSION_NT_LATEST) #tells its latest version, not needed as done by __init__(), but be explicit

    def doSetLogger(self,payload):
        (b,err) = self.reader.readPayload(cSETLOGGER_V3_FRAMELEN)
        if err or self.crcError(b): self.error |= cNTDATAFRAME_SETLOGERROR #; return !!IT MUST NOT BE REJECTED FOR SETLOG!!
        payload = self.decode(b, cSETLOGGER_V3_DATALEN, cSETLOGGER_V3_HIGHBITSLEN)
        #if payload!=None and not self.checkReaderError(): !!IT MUST NOT BE REJECTED FOR SETLOG!!!
        #self.unpackSetLogger(payload)
        #self.setLogger( self.setLoggerStruct_V3_NTbus.unpack(payload) )
        (self.TimeStamp32,
         self.Imu1received,self.Imu1done,self.PIDdone,self.Motorsdone,
         self.Imu2done,self.Logdone,self.Loopdone,
         self.State,self.Status,self.Status2,self.ErrorCnt,self.Voltage,
         self.Imu1AnglePitch,self.Imu1AngleRoll,self.Imu1AngleYaw,
         self.Imu2AnglePitch,self.Imu2AngleRoll,self.Imu2AngleYaw,
         self.highres1,self.highres2,self.highres3,
        ) = self.setLoggerStruct_V3_NTbus.unpack(payload)
        self.Imu1AnglePitch = self.Imu1AnglePitch*16 + (self.highres1 & 0x0f)
        self.Imu1AngleRoll = self.Imu1AngleRoll*16 + (self.highres2 & 0x0f)
        self.Imu1AngleYaw = self.Imu1AngleYaw*16 + (self.highres3 & 0x0f)
        self.Imu2AnglePitch = self.Imu2AnglePitch*16 + ((self.highres1>>4) & 0x0f)
        self.Imu2AngleRoll = self.Imu2AngleRoll*16 + ((self.highres2>>4) & 0x0f)
        self.Imu2AngleYaw = self.Imu2AngleYaw*16 + ((self.highres3>>4) & 0x0f)
        return True

    def doSetMotorAll(self,payload):
        (b,err) = self.reader.readPayload(cSETMOTORALL_FRAMELEN)
        if err or self.crcError(b,): self.error |= cNTDATAFRAME_SETMOTERROR; return False
        payload = b
        self.unpackSetMotorAll(payload)
        if self.Flags&0x20 > 0:
            #has been decoded already by self.unpackSetMotorAll() !
            self.VmaxPitch = self.MotPitch = self.VmaxRoll = self.MotRoll = self.VmaxYaw = self.MotYaw = 0
        else:
            # p->VmaxPitch <<= 1;//ntbus_buf[1]
            # a = (u16)(ntbus_buf[2]) + ((u16)(ntbus_buf[3]) << 7);
            self.VmaxPitch <<= 1
            self.MotPitch = (self.MotPitch&0x00ff) + ((self.MotPitch&0xff00) >> 1)
            self.VmaxRoll <<= 1
            self.MotRoll = (self.MotRoll&0x00ff) + ((self.MotRoll&0xff00) >> 1)
            self.VmaxYaw <<= 1
            self.MotYaw = (self.MotYaw&0x00ff) + ((self.MotYaw&0xff00) >> 1)
            self.dURawPitch = self.dURawRoll = self.dURawYaw = 0
        return True

    def doSetCamera(self,payload):
        (b,err) = self.reader.readPayload(cSETCAMERA_FRAMELEN)
        if err or self.crcError(b): self.error |= cNTDATAFRAME_CMDERROR; return False
        payload = b
        (self.CameraFlags,self.CameraModel,self.CameraCmd,self.CameraUnused,self.CameraPwm
         ) = self.setCameraStruct_NTBus.unpack(payload)
        ##    if( pwm>0 ) pwm = (pwm-1) * 10 + 1000;
        if self.CameraPwm > 0: self.CameraPwm = (self.CameraPwm-1) * 10 + 1000
        return True

    def doCmdAccGyro1_V2(self,payload):
        (b,err) = self.reader.readPayload(cCMDACCGYRODATA_V2_FRAMELEN)
        if err or self.crcError(b): self.error |= cNTDATAFRAME_CMDERROR; return False
        payload = self.decode(b, cCMDACCGYRODATA_V2_DATALEN, cCMDACCGYRODATA_V2_HIGHBITSLEN)
        self.unpackCmdAccGyro1_V2(payload)
        return True

    def doCmdAccGyro2_V2(self,payload):
        (b,err) = self.reader.readPayload(cCMDACCGYRODATA_V2_FRAMELEN)
        if err or self.crcError(b): self.error |= cNTDATAFRAME_CMDERROR; return False
        payload = self.decode(b, cCMDACCGYRODATA_V2_DATALEN, cCMDACCGYRODATA_V2_HIGHBITSLEN)
        self.unpackCmdAccGyro2_V2(payload)
        return True

    def doCmdAccGyro1Raw_V2(self,payload):
        (b,err) = self.reader.readPayload(cCMDACCGYRORAWDATA_V2_FRAMELEN)
        if err or self.crcError(b): self.error |= cNTDATAFRAME_CMDERROR; return False
        payload = self.decode(b, cCMDACCGYRORAWDATA_V2_DATALEN, cCMDACCGYRORAWDATA_V2_HIGHBITSLEN)
        self.unpackCmdAccGyro1Raw_V2(payload)
        return True

    def doCmdAccGyro2Raw_V2(self,payload):
        (b,err) = self.reader.readPayload(cCMDACCGYRORAWDATA_V2_FRAMELEN)
        if err or self.crcError(b): self.error |= cNTDATAFRAME_CMDERROR; return False
        payload = self.decode(b, cCMDACCGYRORAWDATA_V2_DATALEN, cCMDACCGYRORAWDATA_V2_HIGHBITSLEN)
        self.unpackCmdAccGyro2Raw_V2(payload)
        return True

    def doCmdAccGyro3Raw_V2(self,payload):
        (b,err) = self.reader.readPayload(cCMDACCGYRORAWDATA_V2_FRAMELEN)
        if err or self.crcError(b): self.error |= cNTDATAFRAME_CMDERROR; return False
        payload = self.decode(b, cCMDACCGYRORAWDATA_V2_DATALEN, cCMDACCGYRORAWDATA_V2_HIGHBITSLEN)
        self.unpackCmdAccGyro3Raw_V2(payload)
        return True

    def doCmdEncoder(self,payload):
        (b,err) = self.reader.readPayload(cCMDENCODERDATA_FRAMELEN)
        if err or self.crcError(b): self.error |= cNTDATAFRAME_CMDERROR; return False
        payload = self.decode(b, cCMDENCODERDATA_DATALEN, cCMDENCODERDATA_HIGHBITSLEN)
        self.unpackCmdEncoder(payload)
        return True

    def doCmdPid(self,payload):
        (b,err) = self.reader.readPayload(cCMDPIDDATA_FRAMELEN)
        if err or self.crcError(b): self.error |= cNTDATAFRAME_CMDERROR; return False
        payload = self.decode(b, cCMDPIDDATA_DATALEN, cCMDPIDDATA_HIGHBITSLEN)
        self.unpackCmdPid(payload)
        return True

    def doCmdAhrs1(self,payload):
        (b,err) = self.reader.readPayload(cCMDAHRSDATA_FRAMELEN)
        if err or self.crcError(b): self.error |= cNTDATAFRAME_CMDERROR; return False
        payload = self.decode(b, cCMDAHRSDATA_DATALEN, cCMDAHRSDATA_HIGHBITSLEN)
        self.unpackCmdAhrs1(payload)
        return True
#STL
    def doCmdSTorM32LinkData(self,payload):
        (b,err) = self.reader.readPayload(cCMDSTORM32LINKDATA_FRAMELEN)
        if err or self.crcError(b): self.error |= cNTDATAFRAME_CMDERROR; return False
        payload = self.decode(b, cCMDSTORM32LINKDATA_DATALEN, cCMDSTORM32LINKDATA_HIGHBITSLEN)
        self.unpackCmdSTorM32LinkData(payload)
        return True
    def doCmdDebugData(self,payload):
        (b,err) = self.reader.readPayload(cCMDDEBUGDATA_FRAMELEN)
        if err or self.crcError(b): self.error |= cNTDATAFRAME_CMDERROR; return False
        payload = self.decode(b, cCMDDEBUGDATA_DATALEN, cCMDDEBUGDATA_HIGHBITSLEN)
        self.unpackCmdDebugData(payload)
        return True

    def doCmdAhrs2(self,payload): return False
    def doCmdAccGyro_V1(self,payload): return False
    def doCmdAccGyro1Raw_V1(self,payload): return False
    def doCmdAccGyro2Raw_V1(self,payload): return False
    def doCmdAccGyro3Raw_V1(self,payload): return False
    def doCmdParameter(self,payload): return False

    def readCmdByte(self):
        (b,err) = self.reader.readPayload(1)
        if err: return 255 #hopefully a really invalid CmdByte
        return int(b[0])

    def decode(self,b,datalen,highbitslen): #returns a bytearray of the raw values
        highbits = b[datalen:datalen+highbitslen]
        highbytenr = 0
        bitpos = 0x01
        d = bytearray()
        crc = 0
        for n in range(datalen):
            if bitpos==0x80:
                highbytenr += 1
                bitpos = 0x01
            c = b[n]
            if highbits[highbytenr] & bitpos: c |= 0x80
            d.append(c)
            crc = crc ^ c
            bitpos <<= 1
        return d

    def crcError(self,payload):
        (b,err) = self.reader.readPayload(1)
        if err: return True
        crc = int(b[0])
        crcpayload = 0
        for n in range(len(payload)): crcpayload = crcpayload ^ payload[n]
        if crcpayload != crc: return True
        return False


###############################################################################
# cNTLogParser
# this is the main class to parse a stream of log packets into a cNTDataFrameObject
#-----------------------------------------------------------------------------#
cCMD_RES    = 0x50 #'RES ';
cCMD_SET    = 0x40 #'SET ';
cCMD_GET    = 0x30 #'GET ';
cCMD_TRG    = 0x10 #'TRG ';
cCMD_CMD    = 0x00 #'CMD ';

cID_ALL     = 0  #'ALL  ';
cID_IMU1    = 1  #'IMU1 '
cID_IMU2    = 2  #'IMU2 '
cID_MOTA    = 3  #'MOTA ';
cID_MOTP    = 4  #'MOTP ';
cID_MOTR    = 5  #'MOTR ';
cID_MOTY    = 6  #'MOTY ';
cID_CAMERA  = 7  #'CAM  ';
cID_LOG     = 11 #'LOG  '; 0x0B
cID_IMU3    = 12 #'IMU3 '

cRESALL     = 0x80 + cCMD_RES + cID_ALL  #0xD0
cTRGALL     = 0x80 + cCMD_TRG + cID_ALL  #0x90
cGETIMU1    = 0x80 + cCMD_GET + cID_IMU1 #0xB1
cGETIMU2    = 0x80 + cCMD_GET + cID_IMU2 #0xB2
cGETIMU3    = 0x80 + cCMD_GET + cID_IMU3 #0xBC
cGETMOTP    = 0x80 + cCMD_GET + cID_MOTP #0xB4
cGETMOTR    = 0x80 + cCMD_GET + cID_MOTR #0xB5
cGETMOTY    = 0x80 + cCMD_GET + cID_MOTY #0xB6
cSETMOTA    = 0x80 + cCMD_SET + cID_MOTA #0xC3
cSETCAMERA  = 0x80 + cCMD_SET + cID_CAMERA
cSETLOG     = 0x80 + cCMD_SET + cID_LOG  #0xCB
cCMDLOG     = 0x80 + cCMD_CMD + cID_LOG  #0x8B

cCMDBYTE_AccGyro1RawData_V1 = 32 #CMD LOG  AccGyro1RawData 32 ###DEPRECATED
cCMDBYTE_AccGyro2RawData_V1 = 33 #CMD LOG  AccGyro2RawData 33 ###DEPRECATED
cCMDBYTE_AccGyroData_V1     = 34 #CMD LOG  AccGyroData 34  ###DEPRECATED
cCMDBYTE_PidData            = 35 #CMD LOG  PidData 35
cCMDBYTE_ParameterData      = 36 #CMD LOG  ParameterData 36
cCMDBYTE_Ahrs1Data          = 37 #CMD LOG  Ahrs1Data 37
cCMDBYTE_Ahrs2Data          = 38 #CMD LOG  Ahrs2Data 38
cCMDBYTE_AccGyro3RawData_V1 = 39 #CMD LOG  AccGyro3RawData 39  ###DEPRECATED

cCMDBYTE_AccGyro1RawData_V2 = 40 #CMD LOG  AccGyro1RawData_V2 40
cCMDBYTE_AccGyro2RawData_V2 = 41 #CMD LOG  AccGyro2RawData_V2 41
cCMDBYTE_AccGyro3RawData_V2 = 42 #CMD LOG  AccGyro3RawData_V2 42

cCMDBYTE_AccGyro1Data_V2    = 43 #CMD LOG  AccGyro1Data_V2 43
cCMDBYTE_AccGyro2Data_V2    = 44 #CMD LOG  AccGyro2Data_V2 44

cCMDBYTE_EncoderData        = 45 #CMD LOG  EncoderData 45
#STL
cCMDBYTE_STorM32LinkData    = 46 #CMD LOG  STorm32LinkData 46 0x2E
cCMDBYTE_DebugData          = 127 #CMD LOG  DebugData 127

#XX
cCMDBYTE_SETLOGGERDATETIME  = 115 #CMD LOG  115


#the reader must provide a function
# reader.appendDataFrame(frame)
#
# baseTime allows to shift the start time
class cNTLogParser:

    def __init__(self,_frame,_reader,_baseTime=0):
        self.reader = _reader

        self.frame = _frame
        self.frame.clear()

        #the TimeStamp32 CANNOT be 0, so 0 can also be used instead of -1
        self.startTimeStamp32 = 0 #also allows to detect that a first valid data frame was read
        self.lastTimeStamp32 = 0  #is set by a Log packet
        self.TimeStamp32 = 0 #copy of frame.TimeStamp32 for convenience
        self.setLog_received = False #allows to detect that one valid Log was read

        self.logTime_error = False
        self.setLog_counter, self.setMotAll_counter, self.setCamera_counter = 0,0,0
        self.getImu1_counter, self.getImu2_counter, self.getImu3_counter = 0,0,0
        self.cmdLog32_counter, self.cmdLog33_counter, self.cmdLog34_counter, self.cmdLog35_counter = 0,0,0,0
        self.cmdLog37_counter, self.cmdLog38_counter, self.cmdLog39_counter = 0,0,0
        self.cmdLog40_counter, self.cmdLog41_counter, self.cmdLog42_counter = 0,0,0
        self.cmdLog43_counter, self.cmdLog44_counter = 0,0
        self.cmdLog45_counter = 0
#STL        
        self.cmdLog46_counter = 0
        self.cmdLog127_counter = 0
        
        self.resAll_counter = 0 #this is used to detect a new log, and is cleared by a SetLog

        self.errorCounts = 0
        self.frameCounts = 0

        self.baseTimeStamp32 = _baseTime #allows to shift the time axis

#XX        
        self.cmdSETLOGGERDATETIME = []

    def clearForNextDataFrame(self):
        self.frame.clear()
        self.logTime_error = False
        self.setLog_counter, self.setMotAll_counter, self.setCamera_counter = 0,0,0
        self.getImu1_counter, self.getImu2_counter, self.getImu3_counter = 0,0,0
        self.cmdLog32_counter, self.cmdLog33_counter, self.cmdLog34_counter, self.cmdLog35_counter = 0,0,0,0
        self.cmdLog37_counter, self.cmdLog38_counter, self.cmdLog39_counter = 0,0,0
        self.cmdLog40_counter, self.cmdLog41_counter, self.cmdLog42_counter = 0,0,0
        self.cmdLog43_counter, self.cmdLog44_counter = 0,0
        self.cmdLog45_counter = 0
#STL        
        self.cmdLog46_counter = 0
        self.cmdLog127_counter = 0 #NTBUS_CMD_DEBUGDATA
       

    #------------------------------------------
    #get data from reader, and parse into a cNTLogDataFrameObhject()
    def parse(self,cmdid,cmdbyte=None,payload=None): #cmdid = 0x80 + cmd + idbyte
        if cmdid < 128: 
            return False #this can't be a cmdid, so skip out fast, and tell caller that the parser did so
        if cmdid==cRESALL: #0x50 # 'RES ';
            self.clearForNextDataFrame();
            self.setLog_received = False
            self.resAll_counter += 1 #this is reset by a SetLog

        elif cmdid==cTRGALL: #'TRG ';'ALL  ';
            pass

        elif cmdid==cSETMOTA: #3 #'SET ';'MOTA ';
            self.frame.doSetMotorAll(payload)
            self.setMotAll_counter += 1

        elif cmdid==cSETCAMERA: #3 #'SET ';'CAM  ';
            self.frame.doSetCamera(payload)
            self.setCamera_counter += 1

        elif cmdid==cSETLOG: #11 #'SET ';'LOG  ';
            self.lastTimeStamp32 = self.TimeStamp32

            self.frame.doSetLogger(payload)

            self.TimeStamp32 = self.frame.TimeStamp32 #keep a copy for convenience

            if self.startTimeStamp32<=0:
                self.startTimeStamp32 = self.TimeStamp32

            #check for a new log in the log file
            if self.resAll_counter == 2 and self.TimeStamp32<self.lastTimeStamp32 and self.TimeStamp32<100000: ##5000:
                self.baseTimeStamp32 += self.lastTimeStamp32 + 1000000 #gap of 1sec
            #if a new log is detected, don't throw an error
            elif self.lastTimeStamp32 > 0 and abs(self.TimeStamp32-self.lastTimeStamp32) > 1700:
                self.logTime_error = True

            self.setLog_counter += 1
            self.resAll_counter = 0
            self.setLog_received = True

        elif cmdid==cGETIMU1: #0x30 #'GET ';
            self.getImu1_counter += 1
        elif cmdid==cGETIMU2:
            self.getImu2_counter += 1
        elif cmdid==cGETIMU3:
            self.getImu3_counter += 1

        elif cmdid==cGETMOTP:
            pass
        elif cmdid==cGETMOTR:
            pass
        elif cmdid==cGETMOTY:
            pass

        elif cmdid==cCMDLOG:
            if cmdbyte==None:
                cmdbyte = self.frame.readCmdByte()
            if cmdbyte==255:
                pass
            elif cmdbyte==cCMDBYTE_PidData: #CMD LOG  PidData 35
                self.frame.doCmdPid(payload)
                self.cmdLog35_counter += 1
            elif cmdbyte==cCMDBYTE_Ahrs1Data:#CMD LOG  Ahrs1Data 37
                self.frame.doCmdAhrs1(payload)
                self.cmdLog37_counter += 1
            elif cmdbyte==cCMDBYTE_Ahrs2Data: #no. 38
                self.frame.doCmdAhrs2(payload)
                self.cmdLog38_counter += 1
            elif cmdbyte==cCMDBYTE_ParameterData: #CMD LOG  ParameterData 36
                self.frame.doCmdParameter(payload)
            #new V2 commands
            elif cmdbyte==cCMDBYTE_AccGyro1RawData_V2: #CMD LOG  AccGyro1RawData_V2 40
                self.frame.doCmdAccGyro1Raw_V2(payload)
                self.cmdLog40_counter += 1
            elif cmdbyte==cCMDBYTE_AccGyro2RawData_V2: #CMD LOG  AccGyro2RawData_V2 41
                self.frame.doCmdAccGyro2Raw_V2(payload)
                self.cmdLog41_counter += 1
            elif cmdbyte==cCMDBYTE_AccGyro3RawData_V2: #CMD LOG  AccGyro3RawData_V2 42
                self.frame.doCmdAccGyro3Raw_V2(payload)
                self.cmdLog42_counter += 1
            elif cmdbyte==cCMDBYTE_AccGyro1Data_V2: #CMD LOG  AccGyro1Data_V2 43
                self.frame.doCmdAccGyro1_V2(payload)
                self.cmdLog43_counter += 1
            elif cmdbyte==cCMDBYTE_AccGyro2Data_V2: #CMD LOG  AccGyro2Data_V2 44
                self.frame.doCmdAccGyro2_V2(payload)
                self.cmdLog44_counter += 1
            elif cmdbyte==cCMDBYTE_EncoderData: #no. 45
                self.frame.doCmdEncoder(payload)
                self.cmdLog45_counter += 1
#STL                
            elif cmdbyte==cCMDBYTE_STorM32LinkData: #no. 46
                self.frame.doCmdSTorM32LinkData(payload)
                self.cmdLog46_counter += 1
            elif cmdbyte==cCMDBYTE_DebugData: #no. 127
                self.frame.doCmdDebugData(payload)
                self.cmdLog127_counter += 1
            #deprectaed V1 commands
            elif cmdbyte==cCMDBYTE_AccGyro1RawData_V1: #no. 32
                self.frame.doCmdAccGyro1Raw_V1(payload)
                self.cmdLog32_counter += 1
            elif cmdbyte==cCMDBYTE_AccGyro2RawData_V1: #no. 33
                self.frame.doCmdAccGyro2Raw_V1(payload)
                self.cmdLog33_counter += 1
            elif cmdbyte==cCMDBYTE_AccGyro3RawData_V1: #no. 39
                self.frame.doCmdAccGyro3Raw_V1(payload)
                self.cmdLog39_counter += 1
            elif cmdbyte==cCMDBYTE_AccGyroData_V1: #no. 34
                self.frame.doCmdAccGyro_V1(payload)
                self.cmdLog34_counter += 1
#XX
            elif cmdbyte==cCMDBYTE_SETLOGGERDATETIME: #no. 115
                self.cmdSETLOGGERDATETIME.append('SETLOGGERDATETIME cmd at time '+str(self.frame.Time/1000)+' ms')
                
        return True #parser did his job, so tell caller

    #------------------------------------------
    #analyzes the received frame, and appends it if correct
    # stxerror allows the caller to have additional errors considered
    # returns error
    def analyzeAndAppend(self,cmdid,stxerror): #cmdid = 0x80 + cmd + idbyte
        frameError = False
        if cmdid==cTRGALL:

            if self.setLog_received: #a SetLog had been received before, so this is the 2nd TrgAll
                if stxerror: frameError = True
                elif self.frame.error&cNTDATAFRAME_SETLOGERROR > 0: frameError = True
                elif self.frame.error&cNTDATAFRAME_SETMOTERROR > 0: frameError = True
                elif self.logTime_error: frameError = True
                elif self.setLog_counter != 1: frameError = True
                elif self.getImu1_counter != 1: frameError = True
                elif self.getImu2_counter > 1: frameError = True #was !=1
                elif self.getImu3_counter > 1: frameError = True
                elif self.setMotAll_counter != 1: frameError = True
                elif self.setCamera_counter > 1: frameError = True

                elif self.cmdLog32_counter > 1: frameError = True
                elif self.cmdLog33_counter > 1: frameError = True
                elif self.cmdLog34_counter > 1: frameError = True
                elif self.cmdLog35_counter > 1: frameError = True
                elif self.cmdLog37_counter > 1: frameError = True
                elif self.cmdLog38_counter > 1: frameError = True
                elif self.cmdLog39_counter > 1: frameError = True
                elif self.cmdLog40_counter > 1: frameError = True
                elif self.cmdLog41_counter > 1: frameError = True
                elif self.cmdLog42_counter > 1: frameError = True
                elif self.cmdLog43_counter > 1: frameError = True
                elif self.cmdLog44_counter > 1: frameError = True
                elif self.cmdLog45_counter > 1: frameError = True
#STL
                elif self.cmdLog46_counter > 1: frameError = True
                elif self.cmdLog127_counter > 1: frameError = True

                if not frameError:
                    self.frame.calculateTime( self.startTimeStamp32-self.baseTimeStamp32 )
                    self.frame.calculateInjectedValues()
                    # this allows appendDataFrame() to do some additional error checks
                    if self.reader.appendDataFrame(self.frame):
                        frameError = True
                    else:
                        self.frameCounts += 1

            self.clearForNextDataFrame()

        if frameError:
            self.errorCounts += 1
        return frameError


###############################################################################
# cNTLogFileReader
# this is the main class to read in a NTLogger data log file
# it generates a number of data lists, for easier handling in the GUI
#-----------------------------------------------------------------------------#
class cNTLogFileReader:

    def __init__(self):
        self.logVersion = cLOGVERSION_UNINITIALIZED #was a BUG, right? cLOGTYPE_UNINITIALIZED #private

        
    def readLogFile(self,loadLogThread,fileName,createTraffic):
        try:
            F = open(fileName, 'rb')
        except:
            return '','',''

        #this is the header which preludes each data packet, 1+1+4+1+1+1 = 9 bytes, stx = 'R'
        # R(u8) size(u8)(total packet) timestamp(u32) cmd(u8) id(u8) cmdbyte(u8) payload(size-9) (there is no crc)
        headerStruct = struct.Struct('=BBIBBB')
        stx,size,timestamp,cmd,idbyte,cmdbyte = 0,0,0,0,0,0

        frame = cNTLogFileDataFrame()
        parser = cNTLogParser(frame, self)

        logItemList = cNTLogDataItemList()

        trafficlog = []
        
        #need to be self so that appendDataFrame() can be called by the parser
        self.datalog = []
        self.datalog.append( logItemList.getNamesAsStr('\t') + '\n' )
        self.datalog.append( logItemList.getUnitsAsStr('\t') + '\n' )

        self.rawdatalog = []

        trgall_timestamp_last = -1
        trafficlog_counter = 0
        stxerror = False

        byte_counter = 0
        byte_max = QFile(fileName).size()
        byte_percentage = 0
        byte_step = 5

        ##FBytesIO = BytesIO(F.read())  //THIS IS NOT FASTER AT ALL!!
        ##header = FBytesIO.read(9)
        ##payload = FBytesIO.read(size-9)
        frame.setLogVersion(cLOGVERSION_NT_V2) #assume <v0.03 as default
        while 1:
            if( loadLogThread.canceled ): break

            header = F.read(9)
            if header == '' or len(header) != 9:
                break
            byte_counter += 9
            stxerror = False

            #------------------------------------------
            #check log start line
            # there should be a check that this is the first line!!! #XX
            if header[0:1] == b'H' and header[2:] == b'STORM32':
                size = int(header[1])
                restofheader = F.read(size-9) # read the rest of the log start line
                frame.setLogVersion(cLOGVERSION_NT_V3) #v0.03
                header = F.read(9)

            #------------------------------------------
            #Header, read header data into proper fields
            stx, size, timestamp, cmd, idbyte, cmdbyte = headerStruct.unpack(header)
            if size<9:
                break;
            if stx != ord('R'):
                cmd, idbyte, cmdbyte = -1, -1, -1
                stxerror = True #NTbus traffic data frame analyzer
            cmdid = 0x80 + cmd + idbyte

            #------------------------------------------
            #Data, read remaining data into proper fields
            payload = F.read(size-9)
            if payload == '' or len(payload) != size-9:
                break
            byte_counter += size-9

            #------------------------------------------
            #read data send with R cmd
            # merged with traffic data frame analyzer
            parser.parse(cmdid, cmdbyte, payload)

            #------------------------------------------
            #NTbus traffic log
            if( createTraffic or trafficlog_counter<500 ):
                tl = str(trafficlog_counter)
                ts = str(timestamp)
                while len(ts)<10: ts = '0'+ts
                trafficlog.append( tl+'\t'+ts+'  ' )
                
                if cmd==cCMD_RES:   trafficlog.append( 'RES ' )
                elif cmd==cCMD_SET: trafficlog.append( 'SET ' )
                elif cmd==cCMD_GET: trafficlog.append( 'GET ' )
                elif cmd==cCMD_TRG: trafficlog.append( 'TRG ' )
                elif cmd==cCMD_CMD: trafficlog.append( 'CMD ' )
                else: trafficlog.append( '??? ' )

                if idbyte==cID_ALL:
                    trafficlog.append( 'ALL  ' )
                    if cmd==cCMD_TRG:
                        if trgall_timestamp_last >= 0:
                            trafficlog.append( '('+str(timestamp-trgall_timestamp_last)+')' )
                        trgall_timestamp_last = timestamp
                elif idbyte==cID_IMU1: trafficlog.append( 'IMU1 ' )
                elif idbyte==cID_IMU2: trafficlog.append( 'IMU2 ' )
                elif idbyte==cID_MOTA: trafficlog.append( 'MOTA ' )
                elif idbyte==cID_MOTP: trafficlog.append( 'MOTP ' )
                elif idbyte==cID_MOTR: trafficlog.append( 'MOTR ' )
                elif idbyte==cID_MOTY: trafficlog.append( 'MOTY ' )
                elif idbyte==cID_CAMERA: trafficlog.append( 'CAM  ' )
                elif idbyte==cID_LOG:  trafficlog.append( 'LOG  ' )
                elif idbyte==cID_IMU3: trafficlog.append( 'IMU3 ' )
                else: trafficlog.append( '???  ' )

                if stx != ord('R'):
                    trafficlog.append( '\n*******************   ERROR: invalid stx   ****************************************************' )
                elif cmd==cCMD_RES:
                    pass
                elif cmd==cCMD_SET:
                    if idbyte==cID_MOTA:
                        trafficlog.append( '0x'+'{:02X}'.format(frame.Flags) )
                        trafficlog.append( ' '+str(frame.VmaxPitch)+' '+str(frame.MotPitch) )
                        trafficlog.append( ' '+str(frame.VmaxRoll)+' '+str(frame.MotRoll) )
                        trafficlog.append( ' '+str(frame.VmaxYaw)+' '+str(frame.MotYaw) )
                    elif idbyte==cID_CAMERA:
                        trafficlog.append( ' '+str(frame.CameraCmd)+' '+str(frame.CameraPwm) )
                    elif idbyte==cID_LOG:
                        trafficlog.append( str(parser.TimeStamp32) )
                        if parser.TimeStamp32 > 0:
                            trafficlog.append( ' ('+str(parser.TimeStamp32-parser.lastTimeStamp32)+')' )
                            trafficlog.append( ' '+str((parser.TimeStamp32-parser.startTimeStamp32)/1000)+' ms' )
                elif cmd==cCMD_GET:
                    pass
                elif cmd==cCMD_TRG:
                    pass
                elif cmd==cCMD_CMD:
                    trafficlog.append( str(cmdbyte) )
                    if cmdbyte==cCMDBYTE_ParameterData:
                        if frame.ParameterAdr==65535:
                            trafficlog.append( '\t'+str(frame.ParameterNameStr, "utf-8") )
                        else:
                            trafficlog.append( '\t'+str(frame.ParameterAdr) )
                            trafficlog.append( '\t'+str(frame.ParameterNameStr.replace(b'\0',b' '), "utf-8") )
                            trafficlog.append( '\t'+str(frame.ParameterValue) )
#XX    #XX                        
                    if cmdbyte==cCMDBYTE_SETLOGGERDATETIME:
                        trafficlog.append( '\t!!!!!!!!!!!!!!!!!!!!   RTC WRITE   !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!' )

                trafficlog.append( '\n' )
                trafficlog_counter += 1

            #------------------------------------------
            #NTbus traffic data frame analyzer
            frameError = parser.analyzeAndAppend(cmdid, stxerror)

            if frameError:
                if( createTraffic or trafficlog_counter<500 ):
                    trafficlog.append( '*******************   ERROR: lost frame(s)   ****************************************************\n' )

            if 95*(byte_counter/byte_max) > byte_percentage:
                loadLogThread.emitProgress(byte_percentage)
                byte_percentage += byte_step

        #end of while 1:
        F.close();
        loadLogThread.emitProgress(95)
        if not (createTraffic or trafficlog_counter<500 ): trafficlog.append( '...\n' )
        trafficlog.append( 'FRAME COUNTS: '+str(parser.frameCounts) + '\n' )
#Xx        trafficlog.append( 'ERROR COUNTS: '+str(parser.errorCounts) )
#XX        this need NTLogger v0.37xx
        trafficlog.append( 'ERROR COUNTS: '+str(parser.errorCounts) + '\n' )    
        trafficlog.append( 'RTC WRITE COUNTS: '+str(len(parser.cmdSETLOGGERDATETIME))+ '\n' )
        for sss in parser.cmdSETLOGGERDATETIME: 
            trafficlog.append( '  '+sss+'\n' )
        
        if( loadLogThread.canceled ):
            trafficlog = []
            self.datalog = []
            self.rawdatalog = []
        self.logVersion = frame.getLogVersion()
        return trafficlog, self.datalog, self.rawdatalog

    #this is called by the parser
    # returns a bool, True if error occured
    def appendDataFrame(self,_frame):
        self.datalog.append( _frame.getDataLine() )
        self.rawdatalog.append( _frame.getRawDataLine() )
        return False

    def getLogVersion(self):
        return self.logVersion
